/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import java.util.AbstractList;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.TreeMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockCollection;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.HeartbeatManager;
import org.apache.hadoop.hdfs.server.blockmanagement.NumberReplicas;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.hdfs.util.CyclicIteration;
import org.apache.hadoop.hdfs.util.LightWeightLinkedSet;
import org.apache.hadoop.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.shaded.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.hadoop.util.ChunkedArrayList;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class DatanodeAdminManager {
    private static final Logger LOG = LoggerFactory.getLogger(DatanodeAdminManager.class);
    private final Namesystem namesystem;
    private final BlockManager blockManager;
    private final HeartbeatManager hbManager;
    private final ScheduledExecutorService executor;
    private final TreeMap<DatanodeDescriptor, AbstractList<BlockInfo>> outOfServiceNodeBlocks;
    private final Queue<DatanodeDescriptor> pendingNodes;
    private Monitor monitor = null;

    DatanodeAdminManager(Namesystem namesystem, BlockManager blockManager, HeartbeatManager hbManager) {
        this.namesystem = namesystem;
        this.blockManager = blockManager;
        this.hbManager = hbManager;
        this.executor = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("DatanodeAdminMonitor-%d").setDaemon(true).build());
        this.outOfServiceNodeBlocks = new TreeMap();
        this.pendingNodes = new ArrayDeque<DatanodeDescriptor>();
    }

    void activate(Configuration conf) {
        int intervalSecs = (int)conf.getTimeDuration("dfs.namenode.decommission.interval", 30L, TimeUnit.SECONDS);
        Preconditions.checkArgument((intervalSecs >= 0 ? 1 : 0) != 0, (Object)"Cannot set a negative value for dfs.namenode.decommission.interval");
        int blocksPerInterval = conf.getInt("dfs.namenode.decommission.blocks.per.interval", 500000);
        String deprecatedKey = "dfs.namenode.decommission.nodes.per.interval";
        String strNodes = conf.get("dfs.namenode.decommission.nodes.per.interval");
        if (strNodes != null) {
            LOG.warn("Deprecated configuration key {} will be ignored.", (Object)"dfs.namenode.decommission.nodes.per.interval");
            LOG.warn("Please update your configuration to use {} instead.", (Object)"dfs.namenode.decommission.blocks.per.interval");
        }
        Preconditions.checkArgument((blocksPerInterval > 0 ? 1 : 0) != 0, (Object)"Must set a positive value for dfs.namenode.decommission.blocks.per.interval");
        int maxConcurrentTrackedNodes = conf.getInt("dfs.namenode.decommission.max.concurrent.tracked.nodes", 100);
        Preconditions.checkArgument((maxConcurrentTrackedNodes >= 0 ? 1 : 0) != 0, (Object)"Cannot set a negative value for dfs.namenode.decommission.max.concurrent.tracked.nodes");
        this.monitor = new Monitor(blocksPerInterval, maxConcurrentTrackedNodes);
        this.executor.scheduleAtFixedRate(this.monitor, intervalSecs, intervalSecs, TimeUnit.SECONDS);
        LOG.debug("Activating DatanodeAdminManager with interval {} seconds, {} max blocks per interval, {} max concurrently tracked nodes.", new Object[]{intervalSecs, blocksPerInterval, maxConcurrentTrackedNodes});
    }

    void close() {
        this.executor.shutdownNow();
        try {
            this.executor.awaitTermination(3000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    @VisibleForTesting
    public void startDecommission(DatanodeDescriptor node) {
        if (!node.isDecommissionInProgress() && !node.isDecommissioned()) {
            this.hbManager.startDecommission(node);
            if (node.isDecommissionInProgress()) {
                for (DatanodeStorageInfo storage : node.getStorageInfos()) {
                    LOG.info("Starting decommission of {} {} with {} blocks", new Object[]{node, storage, storage.numBlocks()});
                }
                node.getLeavingServiceStatus().setStartTime(Time.monotonicNow());
                this.pendingNodes.add(node);
            }
        } else {
            LOG.trace("startDecommission: Node {} in {}, nothing to do.", (Object)node, (Object)node.getAdminState());
        }
    }

    @VisibleForTesting
    public void stopDecommission(DatanodeDescriptor node) {
        if (node.isDecommissionInProgress() || node.isDecommissioned()) {
            this.hbManager.stopDecommission(node);
            if (node.isAlive()) {
                this.blockManager.processExtraRedundancyBlocksOnInService(node);
            }
            this.pendingNodes.remove((Object)node);
            this.outOfServiceNodeBlocks.remove((Object)node);
        } else {
            LOG.trace("stopDecommission: Node {} in {}, nothing to do.", (Object)node, (Object)node.getAdminState());
        }
    }

    @VisibleForTesting
    public void startMaintenance(DatanodeDescriptor node, long maintenanceExpireTimeInMS) {
        node.setMaintenanceExpireTimeInMS(maintenanceExpireTimeInMS);
        if (!node.isMaintenance()) {
            this.hbManager.startMaintenance(node);
            if (node.isEnteringMaintenance()) {
                for (DatanodeStorageInfo storage : node.getStorageInfos()) {
                    LOG.info("Starting maintenance of {} {} with {} blocks", new Object[]{node, storage, storage.numBlocks()});
                }
                node.getLeavingServiceStatus().setStartTime(Time.monotonicNow());
            }
            this.pendingNodes.add(node);
        } else {
            LOG.trace("startMaintenance: Node {} in {}, nothing to do.", (Object)node, (Object)node.getAdminState());
        }
    }

    @VisibleForTesting
    public void stopMaintenance(DatanodeDescriptor node) {
        if (node.isMaintenance()) {
            this.hbManager.stopMaintenance(node);
            if (!node.isAlive()) {
                this.blockManager.removeBlocksAssociatedTo(node);
            } else {
                this.blockManager.processExtraRedundancyBlocksOnInService(node);
            }
            this.pendingNodes.remove((Object)node);
            this.outOfServiceNodeBlocks.remove((Object)node);
        } else {
            LOG.trace("stopMaintenance: Node {} in {}, nothing to do.", (Object)node, (Object)node.getAdminState());
        }
    }

    private void setDecommissioned(DatanodeDescriptor dn) {
        dn.setDecommissioned();
        LOG.info("Decommissioning complete for node {}", (Object)dn);
    }

    private void setInMaintenance(DatanodeDescriptor dn) {
        dn.setInMaintenance();
        LOG.info("Node {} has entered maintenance mode.", (Object)dn);
    }

    private boolean isSufficient(BlockInfo block, BlockCollection bc, NumberReplicas numberReplicas, boolean isDecommission, boolean isMaintenance) {
        if (this.blockManager.hasEnoughEffectiveReplicas(block, numberReplicas, 0)) {
            LOG.trace("Block {} does not need replication.", (Object)block);
            return true;
        }
        short numExpected = this.blockManager.getExpectedLiveRedundancyNum(block, numberReplicas);
        int numLive = numberReplicas.liveReplicas();
        LOG.trace("Block {} numExpected={}, numLive={}", new Object[]{block, (int)numExpected, numLive});
        if (isDecommission && numExpected > numLive) {
            if (bc.isUnderConstruction() && block.equals((Object)bc.getLastBlock())) {
                if (this.blockManager.hasMinStorage(block, numLive)) {
                    LOG.trace("UC block {} sufficiently-replicated since numLive ({}) >= minR ({})", new Object[]{block, numLive, this.blockManager.getMinStorageNum(block)});
                    return true;
                }
                LOG.trace("UC block {} insufficiently-replicated since numLive ({}) < minR ({})", new Object[]{block, numLive, this.blockManager.getMinStorageNum(block)});
            } else if (numLive >= this.blockManager.getDefaultStorageNum(block)) {
                return true;
            }
        }
        return isMaintenance && numLive >= this.blockManager.getMinReplicationToBeInMaintenance();
    }

    private void logBlockReplicationInfo(BlockInfo block, BlockCollection bc, DatanodeDescriptor srcNode, NumberReplicas num, Iterable<DatanodeStorageInfo> storages) {
        if (!NameNode.blockStateChangeLog.isInfoEnabled()) {
            return;
        }
        int curReplicas = num.liveReplicas();
        short curExpectedRedundancy = this.blockManager.getExpectedRedundancyNum(block);
        StringBuilder nodeList = new StringBuilder();
        for (DatanodeStorageInfo storage : storages) {
            DatanodeDescriptor node = storage.getDatanodeDescriptor();
            nodeList.append((Object)node);
            nodeList.append(' ');
        }
        NameNode.blockStateChangeLog.info("Block: " + (Object)((Object)block) + ", Expected Replicas: " + curExpectedRedundancy + ", live replicas: " + curReplicas + ", corrupt replicas: " + num.corruptReplicas() + ", decommissioned replicas: " + num.decommissioned() + ", decommissioning replicas: " + num.decommissioning() + ", maintenance replicas: " + num.maintenanceReplicas() + ", live entering maintenance replicas: " + num.liveEnteringMaintenanceReplicas() + ", excess replicas: " + num.excessReplicas() + ", Is Open File: " + bc.isUnderConstruction() + ", Datanodes having this block: " + nodeList + ", Current Datanode: " + (Object)((Object)srcNode) + ", Is current datanode decommissioning: " + srcNode.isDecommissionInProgress() + ", Is current datanode entering maintenance: " + srcNode.isEnteringMaintenance());
    }

    @VisibleForTesting
    public int getNumPendingNodes() {
        return this.pendingNodes.size();
    }

    @VisibleForTesting
    public int getNumTrackedNodes() {
        return this.outOfServiceNodeBlocks.size();
    }

    @VisibleForTesting
    public int getNumNodesChecked() {
        return this.monitor.numNodesChecked;
    }

    @VisibleForTesting
    void runMonitorForTest() throws ExecutionException, InterruptedException {
        this.executor.submit(this.monitor).get();
    }

    private class Monitor
    implements Runnable {
        private final int numBlocksPerCheck;
        private final int maxConcurrentTrackedNodes;
        private int numBlocksChecked = 0;
        private int numBlocksCheckedPerLock = 0;
        private int numNodesChecked = 0;
        private DatanodeDescriptor iterkey = new DatanodeDescriptor(new DatanodeID("", "", "", 0, 0, 0, 0));

        Monitor(int numBlocksPerCheck, int maxConcurrentTrackedNodes) {
            this.numBlocksPerCheck = numBlocksPerCheck;
            this.maxConcurrentTrackedNodes = maxConcurrentTrackedNodes;
        }

        private boolean exceededNumBlocksPerCheck() {
            LOG.trace("Processed {} blocks so far this tick", (Object)this.numBlocksChecked);
            return this.numBlocksChecked >= this.numBlocksPerCheck;
        }

        @Override
        public void run() {
            LOG.debug("DatanodeAdminMonitor is running.");
            if (!DatanodeAdminManager.this.namesystem.isRunning()) {
                LOG.info("Namesystem is not running, skipping decommissioning/maintenance checks.");
                return;
            }
            this.numBlocksChecked = 0;
            this.numBlocksCheckedPerLock = 0;
            this.numNodesChecked = 0;
            DatanodeAdminManager.this.namesystem.writeLock();
            try {
                this.processPendingNodes();
                this.check();
            }
            catch (Exception e) {
                LOG.warn("DatanodeAdminMonitor caught exception when processing node.", (Throwable)e);
            }
            finally {
                DatanodeAdminManager.this.namesystem.writeUnlock();
            }
            if (this.numBlocksChecked + this.numNodesChecked > 0) {
                LOG.info("Checked {} blocks and {} nodes this tick", (Object)this.numBlocksChecked, (Object)this.numNodesChecked);
            }
        }

        private void processPendingNodes() {
            while (!(DatanodeAdminManager.this.pendingNodes.isEmpty() || this.maxConcurrentTrackedNodes != 0 && DatanodeAdminManager.this.outOfServiceNodeBlocks.size() >= this.maxConcurrentTrackedNodes)) {
                DatanodeAdminManager.this.outOfServiceNodeBlocks.put(DatanodeAdminManager.this.pendingNodes.poll(), null);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void check() {
            Iterator it = new CyclicIteration(DatanodeAdminManager.this.outOfServiceNodeBlocks, this.iterkey).iterator();
            ArrayList<DatanodeDescriptor> toRemove = new ArrayList<DatanodeDescriptor>();
            while (it.hasNext() && !this.exceededNumBlocksPerCheck() && DatanodeAdminManager.this.namesystem.isRunning()) {
                ++this.numNodesChecked;
                Map.Entry entry = it.next();
                DatanodeDescriptor dn = entry.getKey();
                try {
                    AbstractList<BlockInfo> blocks = (AbstractList<BlockInfo>)entry.getValue();
                    boolean fullScan = false;
                    if (dn.isMaintenance() && dn.maintenanceExpired()) {
                        DatanodeAdminManager.this.stopMaintenance(dn);
                        toRemove.add(dn);
                        continue;
                    }
                    if (dn.isInMaintenance()) continue;
                    if (blocks == null) {
                        LOG.debug("Newly-added node {}, doing full scan to find insufficiently-replicated blocks.", (Object)dn);
                        blocks = this.handleInsufficientlyStored(dn);
                        DatanodeAdminManager.this.outOfServiceNodeBlocks.put(dn, blocks);
                        fullScan = true;
                    } else {
                        LOG.debug("Processing {} node {}", (Object)dn.getAdminState(), (Object)dn);
                        this.pruneReliableBlocks(dn, blocks);
                    }
                    if (blocks.size() == 0) {
                        if (!fullScan) {
                            LOG.debug("Node {} has finished replicating current set of blocks, checking with the full block map.", (Object)dn);
                            blocks = this.handleInsufficientlyStored(dn);
                            DatanodeAdminManager.this.outOfServiceNodeBlocks.put(dn, blocks);
                        }
                        boolean isHealthy = DatanodeAdminManager.this.blockManager.isNodeHealthyForDecommissionOrMaintenance(dn);
                        if (blocks.size() == 0 && isHealthy) {
                            if (dn.isDecommissionInProgress()) {
                                DatanodeAdminManager.this.setDecommissioned(dn);
                                toRemove.add(dn);
                            } else if (dn.isEnteringMaintenance()) {
                                DatanodeAdminManager.this.setInMaintenance(dn);
                            } else {
                                Preconditions.checkState((boolean)false, (String)"Node %s is in an invalid state! Invalid state: %s %s blocks are on this dn.", (Object)((Object)dn), (Object)dn.getAdminState(), (Object)blocks.size());
                            }
                            LOG.debug("Node {} is sufficiently replicated and healthy, marked as {}.", (Object)dn, (Object)dn.getAdminState());
                            continue;
                        }
                        LOG.debug("Node {} {} healthy. It needs to replicate {} more blocks. {} is still in progress.", new Object[]{dn, isHealthy ? "is" : "isn't", blocks.size(), dn.getAdminState()});
                        continue;
                    }
                    LOG.debug("Node {} still has {} blocks to replicate before it is a candidate to finish {}.", new Object[]{dn, blocks.size(), dn.getAdminState()});
                }
                catch (Exception e) {
                    LOG.warn("DatanodeAdminMonitor caught exception when processing node {}.", (Object)dn, (Object)e);
                    DatanodeAdminManager.this.pendingNodes.add(dn);
                    toRemove.add(dn);
                }
                finally {
                    this.iterkey = dn;
                }
            }
            for (DatanodeDescriptor dn : toRemove) {
                Preconditions.checkState((dn.isDecommissioned() || dn.isInService() ? 1 : 0) != 0, (String)"Removing node %s that is not yet decommissioned or in service!", (Object)((Object)dn));
                DatanodeAdminManager.this.outOfServiceNodeBlocks.remove((Object)dn);
            }
        }

        private void pruneReliableBlocks(DatanodeDescriptor datanode, AbstractList<BlockInfo> blocks) {
            this.processBlocksInternal(datanode, blocks.iterator(), null, true);
        }

        private AbstractList<BlockInfo> handleInsufficientlyStored(DatanodeDescriptor datanode) {
            ChunkedArrayList insufficient = new ChunkedArrayList();
            this.processBlocksInternal(datanode, datanode.getBlockIterator(), (List<BlockInfo>)insufficient, false);
            return insufficient;
        }

        private void processBlocksInternal(DatanodeDescriptor datanode, Iterator<BlockInfo> it, List<BlockInfo> insufficientList, boolean pruneReliableBlocks) {
            boolean firstReplicationLog = true;
            int lowRedundancyBlocksInOpenFiles = 0;
            LightWeightLinkedSet<Long> lowRedundancyOpenFiles = new LightWeightLinkedSet<Long>();
            int lowRedundancyBlocks = 0;
            int outOfServiceOnlyReplicas = 0;
            while (it.hasNext()) {
                boolean neededReconstruction;
                if (insufficientList == null && this.numBlocksCheckedPerLock >= this.numBlocksPerCheck) {
                    DatanodeAdminManager.this.namesystem.writeUnlock();
                    try {
                        LOG.debug("Yielded lock during decommission/maintenance check");
                        Thread.sleep(0L, 500);
                    }
                    catch (InterruptedException ignored) {
                        return;
                    }
                    this.numBlocksCheckedPerLock = 0;
                    DatanodeAdminManager.this.namesystem.writeLock();
                }
                ++this.numBlocksChecked;
                ++this.numBlocksCheckedPerLock;
                BlockInfo block = it.next();
                if (((DatanodeAdminManager)DatanodeAdminManager.this).blockManager.blocksMap.getStoredBlock(block) == null) {
                    LOG.trace("Removing unknown block {}", (Object)block);
                    it.remove();
                    continue;
                }
                long bcId = block.getBlockCollectionId();
                if (bcId == -1L) continue;
                BlockCollection bc = DatanodeAdminManager.this.blockManager.getBlockCollection(block);
                NumberReplicas num = DatanodeAdminManager.this.blockManager.countNodes(block);
                int liveReplicas = num.liveReplicas();
                boolean isDecommission = datanode.isDecommissionInProgress();
                boolean isMaintenance = datanode.isEnteringMaintenance();
                boolean bl = neededReconstruction = isDecommission ? DatanodeAdminManager.this.blockManager.isNeededReconstruction(block, num) : DatanodeAdminManager.this.blockManager.isNeededReconstructionForMaintenance(block, num);
                if (neededReconstruction && !((DatanodeAdminManager)DatanodeAdminManager.this).blockManager.neededReconstruction.contains(block) && ((DatanodeAdminManager)DatanodeAdminManager.this).blockManager.pendingReconstruction.getNumReplicas(block) == 0 && DatanodeAdminManager.this.blockManager.isPopulatingReplQueues()) {
                    ((DatanodeAdminManager)DatanodeAdminManager.this).blockManager.neededReconstruction.add(block, liveReplicas, num.readOnlyReplicas(), num.outOfServiceReplicas(), DatanodeAdminManager.this.blockManager.getExpectedRedundancyNum(block));
                }
                if (DatanodeAdminManager.this.isSufficient(block, bc, num, isDecommission, isMaintenance)) {
                    if (!pruneReliableBlocks) continue;
                    it.remove();
                    continue;
                }
                if (insufficientList != null) {
                    insufficientList.add(block);
                }
                if (firstReplicationLog) {
                    DatanodeAdminManager.this.logBlockReplicationInfo(block, bc, datanode, num, ((DatanodeAdminManager)DatanodeAdminManager.this).blockManager.blocksMap.getStorages(block));
                    firstReplicationLog = false;
                }
                ++lowRedundancyBlocks;
                if (bc.isUnderConstruction()) {
                    INode ucFile = DatanodeAdminManager.this.namesystem.getFSDirectory().getInode(bc.getId());
                    if (!(ucFile instanceof INodeFile) || !ucFile.asFile().isUnderConstruction()) {
                        LOG.warn("File {} is not under construction. Skipping add to low redundancy open files!", (Object)ucFile.getLocalName());
                    } else {
                        ++lowRedundancyBlocksInOpenFiles;
                        lowRedundancyOpenFiles.add(ucFile.getId());
                    }
                }
                if (liveReplicas != 0 || num.outOfServiceReplicas() <= 0) continue;
                ++outOfServiceOnlyReplicas;
            }
            datanode.getLeavingServiceStatus().set(lowRedundancyBlocksInOpenFiles, lowRedundancyOpenFiles, lowRedundancyBlocks, outOfServiceOnlyReplicas);
        }
    }
}

