/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.lib.service.hadoop;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.security.PrivilegedExceptionAction;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.lib.server.BaseService;
import org.apache.hadoop.lib.server.ServiceException;
import org.apache.hadoop.lib.service.FileSystemAccess;
import org.apache.hadoop.lib.service.FileSystemAccessException;
import org.apache.hadoop.lib.service.Instrumentation;
import org.apache.hadoop.lib.service.Scheduler;
import org.apache.hadoop.lib.util.Check;
import org.apache.hadoop.lib.util.ConfigurationUtils;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.VersionInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class FileSystemAccessService
extends BaseService
implements FileSystemAccess {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemAccessService.class);
    public static final String PREFIX = "hadoop";
    private static final String INSTRUMENTATION_GROUP = "hadoop";
    public static final String AUTHENTICATION_TYPE = "authentication.type";
    public static final String KERBEROS_KEYTAB = "authentication.kerberos.keytab";
    public static final String KERBEROS_PRINCIPAL = "authentication.kerberos.principal";
    public static final String FS_CACHE_PURGE_FREQUENCY = "filesystem.cache.purge.frequency";
    public static final String FS_CACHE_PURGE_TIMEOUT = "filesystem.cache.purge.timeout";
    public static final String NAME_NODE_WHITELIST = "name.node.whitelist";
    public static final String HADOOP_CONF_DIR = "config.dir";
    private static final String[] HADOOP_CONF_FILES = new String[]{"core-site.xml", "hdfs-site.xml"};
    private static final String FILE_SYSTEM_SERVICE_CREATED = "FileSystemAccessService.created";
    private Collection<String> nameNodeWhitelist;
    Configuration serviceHadoopConf;
    private AtomicInteger unmanagedFileSystems = new AtomicInteger();
    private ConcurrentHashMap<String, CachedFileSystem> fsCache = new ConcurrentHashMap();
    private long purgeTimeout;
    private static final String HTTPFS_FS_USER = "httpfs.fs.user";

    public FileSystemAccessService() {
        super("hadoop");
    }

    @Override
    protected void init() throws ServiceException {
        LOG.info("Using FileSystemAccess JARs version [{}]", (Object)VersionInfo.getVersion());
        String security = this.getServiceConfig().get(AUTHENTICATION_TYPE, "simple").trim();
        if (security.equals("kerberos")) {
            String defaultName = this.getServer().getName();
            String keytab = System.getProperty("user.home") + "/" + defaultName + ".keytab";
            keytab = this.getServiceConfig().get(KERBEROS_KEYTAB, keytab).trim();
            if (keytab.length() == 0) {
                throw new ServiceException(FileSystemAccessException.ERROR.H01, KERBEROS_KEYTAB);
            }
            String principal = defaultName + "/localhost@LOCALHOST";
            principal = this.getServiceConfig().get(KERBEROS_PRINCIPAL, principal).trim();
            if (principal.length() == 0) {
                throw new ServiceException(FileSystemAccessException.ERROR.H01, KERBEROS_PRINCIPAL);
            }
            Configuration conf = new Configuration();
            conf.set("hadoop.security.authentication", "kerberos");
            UserGroupInformation.setConfiguration((Configuration)conf);
            try {
                UserGroupInformation.loginUserFromKeytab((String)principal, (String)keytab);
            }
            catch (IOException ex) {
                throw new ServiceException(FileSystemAccessException.ERROR.H02, ex.getMessage(), ex);
            }
            LOG.info("Using FileSystemAccess Kerberos authentication, principal [{}] keytab [{}]", (Object)principal, (Object)keytab);
        } else if (security.equals("simple")) {
            Configuration conf = new Configuration();
            conf.set("hadoop.security.authentication", "simple");
            UserGroupInformation.setConfiguration((Configuration)conf);
            LOG.info("Using FileSystemAccess simple/pseudo authentication, principal [{}]", (Object)System.getProperty("user.name"));
        } else {
            throw new ServiceException(FileSystemAccessException.ERROR.H09, security);
        }
        String hadoopConfDirProp = this.getServiceConfig().get(HADOOP_CONF_DIR, this.getServer().getConfigDir());
        File hadoopConfDir = new File(hadoopConfDirProp).getAbsoluteFile();
        if (!hadoopConfDir.exists()) {
            hadoopConfDir = new File(this.getServer().getConfigDir()).getAbsoluteFile();
        }
        if (!hadoopConfDir.exists()) {
            throw new ServiceException(FileSystemAccessException.ERROR.H10, hadoopConfDir);
        }
        try {
            this.serviceHadoopConf = this.loadHadoopConf(hadoopConfDir);
        }
        catch (IOException ex) {
            throw new ServiceException(FileSystemAccessException.ERROR.H11, ex.toString(), ex);
        }
        LOG.debug("FileSystemAccess FileSystem configuration:");
        for (Map.Entry entry : this.serviceHadoopConf) {
            LOG.debug("  {} = {}", entry.getKey(), entry.getValue());
        }
        this.setRequiredServiceHadoopConf(this.serviceHadoopConf);
        this.nameNodeWhitelist = this.toLowerCase(this.getServiceConfig().getTrimmedStringCollection(NAME_NODE_WHITELIST));
    }

    private Configuration loadHadoopConf(File dir) throws IOException {
        Configuration hadoopConf = new Configuration(false);
        for (String file : HADOOP_CONF_FILES) {
            File f = new File(dir, file);
            if (!f.exists()) continue;
            hadoopConf.addResource(new Path(f.getAbsolutePath()));
        }
        return hadoopConf;
    }

    @Override
    public void postInit() throws ServiceException {
        super.postInit();
        Instrumentation instrumentation = this.getServer().get(Instrumentation.class);
        instrumentation.addVariable("hadoop", "unmanaged.fs", new Instrumentation.Variable<Integer>(){

            @Override
            public Integer getValue() {
                return FileSystemAccessService.this.unmanagedFileSystems.get();
            }
        });
        instrumentation.addSampler("hadoop", "unmanaged.fs", 60, new Instrumentation.Variable<Long>(){

            @Override
            public Long getValue() {
                return FileSystemAccessService.this.unmanagedFileSystems.get();
            }
        });
        Scheduler scheduler = this.getServer().get(Scheduler.class);
        int purgeInterval = this.getServiceConfig().getInt(FS_CACHE_PURGE_FREQUENCY, 60);
        this.purgeTimeout = this.getServiceConfig().getLong(FS_CACHE_PURGE_TIMEOUT, 60L);
        long l = this.purgeTimeout = this.purgeTimeout > 0L ? this.purgeTimeout : 0L;
        if (this.purgeTimeout > 0L) {
            scheduler.schedule(new FileSystemCachePurger(), (long)purgeInterval, (long)purgeInterval, TimeUnit.SECONDS);
        }
    }

    private Set<String> toLowerCase(Collection<String> collection) {
        HashSet<String> set = new HashSet<String>();
        for (String value : collection) {
            set.add(StringUtils.toLowerCase((String)value));
        }
        return set;
    }

    @Override
    public Class getInterface() {
        return FileSystemAccess.class;
    }

    @Override
    public Class[] getServiceDependencies() {
        return new Class[]{Instrumentation.class, Scheduler.class};
    }

    protected UserGroupInformation getUGI(String user) throws IOException {
        return UserGroupInformation.createProxyUser((String)user, (UserGroupInformation)UserGroupInformation.getLoginUser());
    }

    protected void setRequiredServiceHadoopConf(Configuration conf) {
        conf.set("fs.hdfs.impl.disable.cache", "true");
    }

    protected FileSystem createFileSystem(Configuration namenodeConf) throws IOException {
        CachedFileSystem newCachedFS;
        String user = UserGroupInformation.getCurrentUser().getShortUserName();
        CachedFileSystem cachedFS = this.fsCache.putIfAbsent(user, newCachedFS = new CachedFileSystem(this.purgeTimeout));
        if (cachedFS == null) {
            cachedFS = newCachedFS;
        }
        Configuration conf = new Configuration(namenodeConf);
        conf.set(HTTPFS_FS_USER, user);
        return cachedFS.getFileSystem(conf);
    }

    protected void closeFileSystem(FileSystem fs) throws IOException {
        if (this.fsCache.containsKey(fs.getConf().get(HTTPFS_FS_USER))) {
            this.fsCache.get(fs.getConf().get(HTTPFS_FS_USER)).release();
        }
    }

    protected void validateNamenode(String namenode) throws FileSystemAccessException {
        if (this.nameNodeWhitelist.size() > 0 && !this.nameNodeWhitelist.contains("*") && !this.nameNodeWhitelist.contains(StringUtils.toLowerCase((String)namenode))) {
            throw new FileSystemAccessException(FileSystemAccessException.ERROR.H05, namenode, "not in whitelist");
        }
    }

    protected void checkNameNodeHealth(FileSystem fileSystem) throws FileSystemAccessException {
    }

    @Override
    public <T> T execute(String user, final Configuration conf, final FileSystemAccess.FileSystemExecutor<T> executor) throws FileSystemAccessException {
        Check.notEmpty(user, "user");
        Check.notNull(conf, "conf");
        Check.notNull(executor, "executor");
        if (!conf.getBoolean(FILE_SYSTEM_SERVICE_CREATED, false)) {
            throw new FileSystemAccessException(FileSystemAccessException.ERROR.H04, new Object[0]);
        }
        if (conf.get("fs.defaultFS") == null || conf.getTrimmed("fs.defaultFS").length() == 0) {
            throw new FileSystemAccessException(FileSystemAccessException.ERROR.H06, "fs.defaultFS");
        }
        try {
            this.validateNamenode(new URI(conf.get("fs.defaultFS")).getAuthority());
            UserGroupInformation ugi = this.getUGI(user);
            return (T)ugi.doAs(new PrivilegedExceptionAction<T>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public T run() throws Exception {
                    FileSystem fs = FileSystemAccessService.this.createFileSystem(conf);
                    Instrumentation instrumentation = FileSystemAccessService.this.getServer().get(Instrumentation.class);
                    Instrumentation.Cron cron = instrumentation.createCron();
                    try {
                        FileSystemAccessService.this.checkNameNodeHealth(fs);
                        cron.start();
                        Object t = executor.execute(fs);
                        return t;
                    }
                    finally {
                        cron.stop();
                        instrumentation.addCron("hadoop", executor.getClass().getSimpleName(), cron);
                        FileSystemAccessService.this.closeFileSystem(fs);
                    }
                }
            });
        }
        catch (FileSystemAccessException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new FileSystemAccessException(FileSystemAccessException.ERROR.H03, ex);
        }
    }

    public FileSystem createFileSystemInternal(String user, final Configuration conf) throws IOException, FileSystemAccessException {
        Check.notEmpty(user, "user");
        Check.notNull(conf, "conf");
        if (!conf.getBoolean(FILE_SYSTEM_SERVICE_CREATED, false)) {
            throw new FileSystemAccessException(FileSystemAccessException.ERROR.H04, new Object[0]);
        }
        try {
            this.validateNamenode(new URI(conf.get("fs.defaultFS")).getAuthority());
            UserGroupInformation ugi = this.getUGI(user);
            return (FileSystem)ugi.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<FileSystem>(){

                @Override
                public FileSystem run() throws Exception {
                    return FileSystemAccessService.this.createFileSystem(conf);
                }
            });
        }
        catch (IOException ex) {
            throw ex;
        }
        catch (FileSystemAccessException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new FileSystemAccessException(FileSystemAccessException.ERROR.H08, ex.getMessage(), ex);
        }
    }

    @Override
    public FileSystem createFileSystem(String user, Configuration conf) throws IOException, FileSystemAccessException {
        this.unmanagedFileSystems.incrementAndGet();
        return this.createFileSystemInternal(user, conf);
    }

    @Override
    public void releaseFileSystem(FileSystem fs) throws IOException {
        this.unmanagedFileSystems.decrementAndGet();
        this.closeFileSystem(fs);
    }

    @Override
    public Configuration getFileSystemConfiguration() {
        Configuration conf = new Configuration(true);
        ConfigurationUtils.copy(this.serviceHadoopConf, conf);
        conf.setBoolean(FILE_SYSTEM_SERVICE_CREATED, true);
        conf.set("fs.permissions.umask-mode", "000");
        return conf;
    }

    private class FileSystemCachePurger
    implements Runnable {
        private FileSystemCachePurger() {
        }

        @Override
        public void run() {
            int count = 0;
            for (CachedFileSystem cacheFs : FileSystemAccessService.this.fsCache.values()) {
                try {
                    count += cacheFs.purgeIfIdle() ? 1 : 0;
                }
                catch (Throwable ex) {
                    LOG.warn("Error while purging filesystem, " + ex.toString(), ex);
                }
            }
            LOG.debug("Purged [{}} filesystem instances", (Object)count);
        }
    }

    private static class CachedFileSystem {
        private FileSystem fs;
        private long lastUse;
        private long timeout;
        private int count;

        public CachedFileSystem(long timeout) {
            this.timeout = timeout;
            this.lastUse = -1L;
            this.count = 0;
        }

        synchronized FileSystem getFileSystem(Configuration conf) throws IOException {
            if (this.fs == null) {
                this.fs = FileSystem.get((Configuration)conf);
            }
            this.lastUse = -1L;
            ++this.count;
            return this.fs;
        }

        synchronized void release() throws IOException {
            --this.count;
            if (this.count == 0) {
                if (this.timeout == 0L) {
                    this.fs.close();
                    this.fs = null;
                    this.lastUse = -1L;
                } else {
                    this.lastUse = System.currentTimeMillis();
                }
            }
        }

        synchronized boolean purgeIfIdle() throws IOException {
            boolean ret = false;
            if (this.count == 0 && this.lastUse != -1L && System.currentTimeMillis() - this.lastUse > this.timeout) {
                this.fs.close();
                this.fs = null;
                this.lastUse = -1L;
                ret = true;
            }
            return ret;
        }
    }
}

